//
//  PredictionService.swift
//  death_app Watch App
//
//  Service layer for prediction engine integration
//

import Foundation
import HealthKit
import Combine

@MainActor
class PredictionService: ObservableObject {
    static let shared = PredictionService()
    
    @Published var currentPrediction: LifeExpectancyPrediction?
    @Published var isCalculating = false
    @Published var lastError: Error?
    
    private let predictionEngine = PredictionEngine.shared
    private let healthKitService = HealthKitService()
    private let surveyIntegrationService = SurveyIntegrationService()
    
    private var cancellables = Set<AnyCancellable>()
    
    private init() {
        setupSubscriptions()
    }
    
    /// Calculate life expectancy with current data
    func calculateLifeExpectancy() async {
        isCalculating = true
        lastError = nil
        
        do {
            // Get health data from HealthKit
            let healthSnapshot = try await healthKitService.fetchLatestHealthData()
            
            // Get survey data
            surveyIntegrationService.refreshData()
            guard let surveyData = surveyIntegrationService.currentSurveyData else {
                throw PredictionError.noSurveyData
            }
            
            // Extract demographics from health data
            guard let age = healthSnapshot.age,
                  let biologicalSex = healthSnapshot.biologicalSex else {
                throw PredictionError.insufficientDemographics
            }
            
            let gender: Gender = {
                switch biologicalSex {
                case .male: return .male
                case .female: return .female
                default: return .other
                }
            }()
            
            // Create prediction health data
            let healthData = PredictionHealthData(
                restingHeartRate: healthSnapshot.restingHeartRate ?? 70.0,
                vo2Max: healthSnapshot.vo2Max ?? 40.0,
                bmi: healthSnapshot.bodyMassIndex ?? 25.0,
                averageSleepHours: healthSnapshot.sleepAnalysis?.totalSleepHours ?? 7.5,
                bloodPressureSystolic: nil,
                bloodPressureDiastolic: nil,
                cholesterolTotal: nil,
                glucoseLevel: nil
            )
            
            // Calculate prediction
            let prediction = predictionEngine.calculateLifeExpectancy(
                age: age,
                gender: gender,
                healthData: healthData,
                surveyData: surveyData
            )
            
            currentPrediction = prediction
            
            // Log calculation for analytics
            logPredictionCalculation(prediction)
            
        } catch {
            lastError = error
            print("Prediction calculation failed: \(error)")
        }
        
        isCalculating = false
    }
    
    /// Force recalculation (bypass cache)
    func forceRecalculation() async {
        // Clear cache first
        PredictionCache().clearCache()
        await calculateLifeExpectancy()
    }
    
    /// Get prediction summary for display
    func getPredictionSummary() -> PredictionSummary? {
        guard let prediction = currentPrediction else { return nil }
        
        return PredictionSummary(
            lifeExpectancy: prediction.lifeExpectancy,
            changeFromPrevious: prediction.changeFromPrevious,
            topRiskFactors: getTopRiskFactors(from: prediction.riskFactors),
            lastUpdated: prediction.calculatedDate
        )
    }
    
    // MARK: - Private Methods
    
    private func setupSubscriptions() {
        // Recalculate when health data changes
        healthKitService.$currentHealthData
            .debounce(for: .seconds(2), scheduler: DispatchQueue.main)
            .sink { [weak self] _ in
                Task { @MainActor in
                    await self?.calculateLifeExpectancy()
                }
            }
            .store(in: &cancellables)
        
        // Recalculate when survey data changes
        surveyIntegrationService.$currentSurveyData
            .debounce(for: .seconds(1), scheduler: DispatchQueue.main)
            .sink { [weak self] _ in
                Task { @MainActor in
                    await self?.calculateLifeExpectancy()
                }
            }
            .store(in: &cancellables)
    }
    
    private func getTopRiskFactors(from breakdown: RiskFactorBreakdown) -> [RiskFactorImpact] {
        let factors = [
            RiskFactorImpact(factor: "Heart Rate", yearsImpact: (breakdown.heartRate - 1.0) * 5.0, hazardRatio: breakdown.heartRate),
            RiskFactorImpact(factor: "VO2 Max", yearsImpact: (breakdown.vo2Max - 1.0) * 3.0, hazardRatio: breakdown.vo2Max),
            RiskFactorImpact(factor: "Smoking", yearsImpact: (breakdown.smoking - 1.0) * 8.0, hazardRatio: breakdown.smoking),
            RiskFactorImpact(factor: "BMI", yearsImpact: (breakdown.bmi - 1.0) * 4.0, hazardRatio: breakdown.bmi),
            RiskFactorImpact(factor: "Sleep", yearsImpact: (breakdown.sleep - 1.0) * 2.0, hazardRatio: breakdown.sleep)
        ]
        
        return factors
            .filter { abs($0.hazardRatio - 1.0) > 0.05 } // Only significant factors
            .sorted { abs($0.hazardRatio - 1.0) > abs($1.hazardRatio - 1.0) } // Sort by impact magnitude
            .prefix(3) // Top 3 factors
            .map { $0 }
    }
    
    private func logPredictionCalculation(_ prediction: LifeExpectancyPrediction) {
        // In production, would log to analytics service
        print("Prediction calculated: \(prediction.lifeExpectancy) years")
        if let change = prediction.changeFromPrevious {
            print("Change from previous: \(change.formattedChange)")
        }
    }
}

// MARK: - Supporting Types

struct PredictionSummary {
    let lifeExpectancy: Double
    let changeFromPrevious: PredictionChange?
    let topRiskFactors: [RiskFactorImpact]
    let lastUpdated: Date
}

enum PredictionError: Error, LocalizedError {
    case noSurveyData
    case insufficientDemographics
    case healthDataUnavailable
    case calculationFailed
    
    var errorDescription: String? {
        switch self {
        case .noSurveyData:
            return "Survey data is required for life expectancy calculation"
        case .insufficientDemographics:
            return "Age and gender information is required from HealthKit"
        case .healthDataUnavailable:
            return "Health data is not available"
        case .calculationFailed:
            return "Life expectancy calculation failed"
        }
    }
}
